#include <iostream>
#include <stdexcept>

// How do we check for guaranteed out-of-range conditions
// during compile time?

// ---- Template Selection Helper
//
// SFINAE in all it's glory!
// Substitution Failure Is Not An Error
//
// meaning:
//
// If during template instantiation one of the generated
// return- or argument-types is invalid,
// NO compile error is issued
//
// Instead, the template instantiation is just _removed_
// from the overload set; as if we would never have written
// that definition.
//
// If this means nothing to you, don't worry.
// The example below will help.
//

// 'enable_if' is a meta-function which will be used to
// specify that a function must only be considered if
// a specific condition is met.
//
template<bool Condition, class T = void>
struct enable_if {
  typedef T type;
};

// If the condition evaluates to false, we do _not_
// return a public 'type', meaning
// enable_if<false, int>::type
// simply does not exist.
template<class T>
struct enable_if<false, T> {};

// ---- Meta Assertions

#define STATIC_ASSERT(x) static const int STATIC_ASSERT_ ## __COUNT__ [(x)]

// ---- Runtime Error

struct out_of_range : virtual std::out_of_range {
  out_of_range() : std::out_of_range("value out of range") {}
};

// ---- Our Actual 'restricted' type

template<class T, T Min, T Max>
struct restricted {
  STATIC_ASSERT((Min < Max));
  
  T value;
  
  explicit restricted(T o)
    : value(o) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
  
  template<T OMin, T OMax>
  restricted(restricted<T, OMin, OMax> o
    // To enable a function overload based on the truth-value
    // of a meta-expression, using a dummy argument
    // (non-used function argument with default value)
    // is usually the simplest way.
    //
    // How does it work:
    //   if OMin < Max is true:
    //      typename enable_if<...>::type
    //      will result in a valid C++ type, therefore
    //      the function overload can be used.
    //   otherwise
    //       this expression is not valid; the compiler
    //       will continue as if this definition does not exist.
    //
    // Here we want to only allow the constructor if the
    // restricted<> intervals overlap.
    //
    // this object:  ......[oooooo]..... original
    // other ok:     .[oooo]............ overlap left
    // other ok:     .........[o]....... subset
    // other ok:     ...........[ooo]... overlap right
    // fail 1:       .[o]............... too low
    // fail 2:       ...............[o]. too high
    //
    // Let's start by checking the other's minimum value
    // is less than this classes maximum.  In terms
    // of template parameters, 'OMin < Max':
    , typename enable_if< OMin <= Max >::type* = 0
  )
    : value(o.value) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
};

// ---- How to use it

int main() {
  restricted<int, 0, 10> i(5);

  // introducing compile-time checks
  try {
    // META:
    // This fails now:
    restricted<int, -10, -1> iAmToNegativeForMyLife(i);    
    // TODO comment out to continue
    
    // But this still compiles:
    restricted<int, 11, 20> thinkPositive(i);
    std::cerr << "error: out_of_range not thrown" << std::endl;
  } catch(out_of_range const&) {
    std::cerr
      << "!!!\n"
      << "!!! but hey: this should be a compile time check!\n"
      << "!!!" << std::endl;
  }
}
